function [Act_Rate,ChanDelays,MaxAdcRext]=STM32F411_ADC_SETUP(skt,Pins,Rate,RES_VAL,SAMP_CONST)
%
%function [Act_Rate,ChanDelays,MaxAdcRext]=STM32F411_ADC_SETUP(skt,Pins,Rate,RES_VAL,SAMP_CONST)
%
%  Set up an STM32F411 board for data capture with peek poke
%  Note - Data is left allgined in a 16 bit word, i.e. full scale is approx 65535
%  to convert output to volts  vout = num_out *1.22 / channel17 output (calibrate using internal reference)
%
%  skt           : peek poke udp skt structure
%  Pins          : io port pin numbers - 0 to 7 - on PA0-7 8 & 9 on PB0 and PB1
%                  17  gives internal referenc = 1.21V; 18 gives Chip Temp
%  Rate          : per channel conversion rate
%  RES_VAL       : ADC Resolution - default is 12 bits
%     0 = 12 bit
%     1 = 10 bits
%     2 = 8 bits
%     3 = 6 bits
%
%  SAMP_CONST     : Sampling time - default is 4  giving 84 cycles - a over 3uS
%     0-> 3 cycles of ADC Clock (usually 25MHz)
%     1-> 15 cycles
%     2-> 28 cycles
%     3-> 56 cycles
%     4-> 84 cycles
%     5-> 112 cycles
%     6-> 144 cycles
%     7-> 480 cycles
%
%   Act_Rate       : Actual sample rate per channel frame achieved (Hz)
%   ChanDelays     : Vector of dhannel delays relative to 1st channel   (Secs)
%   MaxAdcRext     : Maximum allowable external resistance of inputs to allow adequate ADC settling
%
% [Act_Rate,ChanDelays,MaxAdcRext]=STM32F411_ADC_SETUP(skt,[0 2 3],10000)
%
% See Also STM32F411_ADC_PLOT.m STM32F411_ADC_CAL.m
%
% Ian Stothers 4th March 2021  Version 0
% Updated 19 Sept 2022 - Made rate a integer ratio to adc clock - to avoid sampling jitter
% To Do
% Single shot capture
% ADC Calibration
% some timer bitfield need unhardcoding - search debug
% make ADC_PRE a function of cpu type ?



if (nargin<5)
  SAMP_CONST=4;
endif
if (nargin<4)
  RES_VAL=0;
endif

STM32F411_bitfields; % Get some bitfields

% Read Core Clock rate from registry
FT_Clock=udpget(skt,"SystemCoreClock");

%Configure the relevant gpio pins as analogue
LPins=Pins(find(Pins<8));
if min(size(LPins>0))
 %Make sure peripheral clock is on
 Port='GPIOA';
 udp_bit_set(skt,"RCC_AHB1ENR",[Port "EN"]);
 for pin=LPins
  STM32F411_IO_Config(skt,"GPIOA",pin,3,0,0,0,0,5);
 end
endif

HPins=Pins(find(Pins>7));
HPins=HPins(find(HPins<10));
if min(size(HPins>0))
 %Make sure peripheral clock is on
 Port='GPIOB';
 udp_bit_set(skt,"RCC_AHB1ENR",[Port "EN"]);
 for pin=HPins-8
  STM32F411_IO_Config(skt,"GPIOB",pin,3,0,0,0,0,0);
 end
endif



VRefPin=Pins(find(Pins==17));

Num_ADC_Chans=max(size(Pins));

%Shut down peripherals
%udpset(skt,"TIM2_PSC",0);                 % make sure timer stops quickly
%udpset(skt,"TIM2_ARR",20000);             % make sure timer stops quickly
%udpset(skt,"TIM2_CNT",100);               % make sure timer stops quickly
udp_bit_clear(skt,"ADC1_CR2",ADON);       % tell ADC1 to stop
udp_bit_clear(skt,"DMA2_S0_CR",EN);       % tell dma to stop
udp_bit_clear(skt,"ADC1_CR2",ADON);       % tell ADC1 to stop
udp_bit_clear(skt,"TIM2_CR1",CEN);        % tell the timer to stop
udp_bit_clear(skt,"RCC_AHB1ENR",TIM2EN);  % TIM2 clock is off
udp_bit_clear(skt,"RCC_APB2ENR",ADC1CEN); % ADC1 clock is off
udp_bit_clear(skt,"RCC_AHB1ENR",DMA2EN);  % DMA clock is off

%Set up ADC1
udp_bit_set(skt,"RCC_APB2ENR",ADC1CEN);   % ADC1 clock is on

% Clear the Status Reg
udpset(skt,"ADC1_SR",0);
% ADC1_CR1
SMULT=1;
if (max(size(Pins))==1)
SMULT=0;
end
ADC1_CR1_CLEAR =  OVRIE+ (RES*3)+ AWDEN+ JAWDEN+ (DISCNUM*7)+ JDISCEN + DISCEN + JAUTO + AWDSGL + SCAN + JEOCIE + AWDIE + EOCIE + (31* AWDCH);
ADC1_CR1_SET =  SCAN*SMULT + (RES_VAL*RES);
udp_bit_clear(skt,"ADC1_CR1",ADC1_CR1_CLEAR); % Clear bits
udp_bit_set(skt,"ADC1_CR1",ADC1_CR1_SET); % Set Bits

% ADC1_CR2
ADC1_CR2_CLEAR = SWSTART+ (EXTEN*3) +(EXTSEL*15) + (JSWSTART) + (JEXTEN*3)+ (JEXTSEL*15)+ ALIGN + EOCS+ DDS+ DMA + CONT+ ADON;
%ADC1_CR2_SET =  (EXTEN*1) + (EXTSEL*3)+ ALIGN*0 + DMA+DDS;
ADC1_CR2_SET =  (EXTEN*1) + (EXTSEL*6)+ ALIGN*1 + DMA+DDS;
udp_bit_clear(skt,"ADC1_CR2",ADC1_CR2_CLEAR);
udp_bit_set(skt,"ADC1_CR2",ADC1_CR2_SET);
udp_bit_set(skt,"ADC1_CR2",+ADON);


%ADC1_COMMON_CCR
%0 PCLK2 divided by 2
%1 PCLK2 divided by 4
%2 PCLK2 divided by 6
%3 PCLK2 divided by 8
ADC_PR=1;   %at PCLK2 = 100MHz gives ADC Clock of 25MHz - 36 is max at 3.6 V supply
ADC_PR_DIV=[2 4 6 8](ADC_PR+1);   %at PCLK2 = 100MHz gives ADC Clock of 25MHz - 36 is max at 3.6 V supply

ADC1_COMMON_CCR_CLEAR=(ADC_PRE*3)+VBATE+TVSREFE;
TVSREFE_VAL=0;
if min(size(VRefPin>0))
  TVSREFE_VAL=1;
endif

ADC1_COMMON_CCR_SET=((ADC_PRE*ADC_PR)+(TVSREFE*TVSREFE_VAL));
udp_bit_clear(skt,"ADC1_COMMON_CCR",ADC1_COMMON_CCR_CLEAR);
udp_bit_set(skt,"ADC1_COMMON_CCR",ADC1_COMMON_CCR_SET);

SampCycles=[3 15 28 56 84 112 144 480]; % lookup for SampCycles(SAMP_CONST+1)
% ADC1_SMPR2
MaxAdcRext=((SampCycles(SAMP_CONST+1)-0.5)/((FT_Clock/(2+(2*(ADC_PR))))*7e-12*log(2^(12-(RES_VAL*2)+2))))-6000;
if (MaxAdcRext<1000)
  printf("Warning - ADC Rext must be below %f ohms\n",MaxAdcRext);
endif
cv=sum(ADC_SMP2);
udp_bit_clear(skt,"ADC1_SMPR2",cv*7); % ADC sample time
udp_bit_set(skt,"ADC1_SMPR2",cv*SAMP_CONST); % ADC sample time set to 3 for all channel

% ADC1_SMPR1
cv=sum(ADC_SMP1);
udp_bit_clear(skt,"ADC1_SMPR1",cv*7); % ADC sample time
udp_bit_set(skt,"ADC1_SMPR1",cv*SAMP_CONST); % ADC sample time set to 3 for all channel

% MAX_ADC_RATE
MAX_ADC_RATE=FT_Clock/((3+SampCycles(SAMP_CONST+1)+(15-2*RES_VAL))*Num_ADC_Chans*(2+(2*(ADC_PR))));
if (Rate>MAX_ADC_RATE)
   printf("Sample Rate %f too high - Rate limited to %f\n",Rate,MAX_ADC_RATE  );
   Rate=MAX_ADC_RATE;
endif
ChanDelays=((1:Num_ADC_Chans)-1)*((3+SampCycles(SAMP_CONST+1)+(15-2*RES_VAL))*(2+(2*(ADC_PR))))/FT_Clock;

%ADC1_JOFRx
udp_bit_clear(skt,"ADC1_JOFR1",sum((2.^[0:11 ])));
udp_bit_clear(skt,"ADC1_JOFR2",sum((2.^[0:11 ])));
udp_bit_clear(skt,"ADC1_JOFR3",sum((2.^[0:11 ])));
udp_bit_clear(skt,"ADC1_JOFR4",sum((2.^[0:11 ])));

%ADC1_HTR
udp_bit_set(skt,"ADC1_HTR",sum((2.^[0:11 ])));

%ADC1_LTR
udp_bit_clear(skt,"ADC1_LTR",sum((2.^[0:11 ])));

PinP = zeros(1,16);
PinP(1:max(size(Pins)))=Pins;

%ADC1_SQR3
udp_bit_clear(skt,"ADC1_SQR3",(2.^30)-1);
udp_bit_set(skt,"ADC1_SQR3",sum((PinP(1:6)).*(32.^(0:5))));

%ADC1_SQR2
udp_bit_clear(skt,"ADC1_SQR2",(2.^30)-1);
udp_bit_set(skt,"ADC1_SQR2",sum((PinP(7:12)).*(32.^(0:5))));

%ADC1_SQR1
udp_bit_clear(skt,"ADC1_SQR1",15*SQR_L);   % Set the number of chamnnels to be converted debug - bitfild names
udp_bit_set(skt,"ADC1_SQR1",(Num_ADC_Chans-1)*SQR_L);
udp_bit_set(skt,"ADC1_SQR1",sum((PinP(13:16)).*(32.^(0:3))));

%ADC1_JSQR
udp_bit_clear(skt,"ADC1_JSQR",sum((2.^[0:21 ])));

%Set up DMA2_S0
udp_bit_set(skt,"RCC_AHB1ENR",DMA2EN); % DMA clock is on
DMA_CLEAR_VAL =sum([ CHSEL*7 MBURST*3 PBURST*3 CT DBM PL*3 PINCOS MSIZE*3 PSIZE*3 MINC PINC CIRC DIR*3 PFCTRL TCIE HTIE TEIE DMEIE EN]);
udp_bit_clear(skt,"DMA2_S0_CR",DMA_CLEAR_VAL);

DMA_SET_VAL =sum([  PL*2  MSIZE PSIZE MINC  CIRC  ]);
udp_bit_set(skt,"DMA2_S0_CR",DMA_SET_VAL);

udpset(skt,"DMA2_S0_M0AR",skt.ptr(skt_find_index(skt,"ADC_DMA_Buffer")));
udpset(skt,"DMA2_S0_PAR",skt.ptr(skt_find_index(skt,"ADC1_DR")));
int2send=floor(skt.len(skt_find_index(skt,"ADC_DMA_Buffer"))/(2*Num_ADC_Chans))*Num_ADC_Chans;
udpset(skt,"DMA2_S0_NDTR",int2send);

udpset(skt,"DMA2_S0_FCR",0); % Direct mode DMA- no FIFO

%Set up Timer 2 to drive ADC
udp_bit_set(skt,"RCC_APB1ENR",TIM2EN); % TIMER2 clock is on
udpset(skt,"TIM2_CR1",0);           %Clear TIM2_CR1
udpset(skt,"TIM2_CR2",0);           %Clear TIM2_CR1

udp_bit_set(skt,"TIM2_CR1",ARPE+URS);%configure TIM2_CR1
udpset(skt,"TIM2_CR2",2*MMS);%configure the timer event = reload

%PSC Prescaler and ARR = n+1 dividers
ADC_CLOCK=FT_Clock/ADC_PR_DIV;
divid=ADC_CLOCK/Rate;
t2_psc=(ceil(divid/65536)*ADC_PR_DIV)-1; % ensure that sample clock is an integer multiple of adc clock - to avoid jitter
t2cnt=round(FT_Clock/((t2_psc+1)*Rate))-1;
Act_Rate=FT_Clock/((t2cnt+1)*(t2_psc+1));


udpset(skt,"TIM2_PSC",t2_psc);%configure the timer prescaler
udpset(skt,"TIM2_ARR",t2cnt);%configure the timer reload (period) register

% Let rip
udp_bit_set(skt,"DMA2_S0_CR",EN);     % Tell DMA to go
udp_bit_set(skt,"TIM2_CR1",CEN);      % Tell TIM2 to go
udp_bit_set(skt,"TIM2_EGR",1);
